# =============================================================================
# ADA P1 Meter - Auto Firmware Updater (CH340 / CP210x + auto driver install)
# =============================================================================

$ErrorActionPreference = "Continue"
$ProgressPreference = "SilentlyContinue"

# ---- Settings ----
$RetrySeconds   = 2      # wait time between retries
$MaxWaitSeconds = 90     # max total wait time before exit
$BaudRate       = 921600 # esptool baud

function Log($msg, $color="Gray") {
    Write-Host $msg -ForegroundColor $color
}

function Is-Admin {
    $id = [Security.Principal.WindowsIdentity]::GetCurrent()
    $p  = New-Object Security.Principal.WindowsPrincipal($id)
    return $p.IsInRole([Security.Principal.WindowsBuiltInRole]::Administrator)
}

# --- ADMIN FORCE (Always) ---
if (-not (Is-Admin)) {
    Log "Admin rights required, restarting..." "Yellow"
    $ps = (Get-Process -Id $PID).Path
    $script = $MyInvocation.MyCommand.Path
    Start-Process -FilePath $ps -ArgumentList "-NoProfile -ExecutionPolicy Bypass -File `"$script`"" -Verb RunAs | Out-Null
    exit
}

function Get-AdaPorts {
    # CH340: VID_1A86, CP210x: VID_10C4
    $vidRegex = 'VID_1A86|VID_10C4'

    $devs = Get-CimInstance Win32_PnPEntity -ErrorAction SilentlyContinue |
        Where-Object { $_.PNPDeviceID -match $vidRegex }

    $ports = @()

    foreach ($d in $devs) {
        $regPath = "HKLM:\SYSTEM\CurrentControlSet\Enum\$($d.PNPDeviceID)\Device Parameters"
        $portName = $null

        try {
            $portName = (Get-ItemProperty -Path $regPath -Name "PortName" -ErrorAction Stop).PortName
        } catch {
            continue
        }

        if ($portName -notmatch '^COM\d+$') { continue }

        $type = if ($d.PNPDeviceID -match 'VID_1A86') { 'CH340' } elseif ($d.PNPDeviceID -match 'VID_10C4') { 'CP210x' } else { 'USB-UART' }

        $ports += [PSCustomObject]@{
            Port = $portName
            Name = $d.Name
            Type = $type
            PnpId = $d.PNPDeviceID
        }
    }

    $ports | Sort-Object Port -Unique
}

function Install-Drivers($DriversRoot) {
    $pnputil = Join-Path $env:SystemRoot "System32\pnputil.exe"
    if (-not (Test-Path $pnputil)) { Log "ERROR: pnputil not found" "Red"; return $false }
    if (-not (Test-Path $DriversRoot)) { Log "ERROR: drivers folder not found" "Red"; return $false }

    $infFiles = Get-ChildItem -Path $DriversRoot -Filter "*.inf" -Recurse -File -ErrorAction SilentlyContinue
    if (-not $infFiles -or $infFiles.Count -eq 0) { Log "ERROR: no INF files in drivers folder" "Red"; return $false }

    Log "Installing drivers..." "Yellow"

    $anyOk = $false
    foreach ($inf in $infFiles) {
        # Silent run: discard output, check exit code only
        & $pnputil /add-driver "$($inf.FullName)" /install | Out-Null
        if ($LASTEXITCODE -eq 0) { $anyOk = $true }
    }

    # Rescan devices
    & $pnputil /scan-devices | Out-Null

    if (-not $anyOk) {
        Log "Driver installation ran, but no successful install detected (signature/compatibility issue possible)." "Yellow"
        # Proceeding anyway as driver might already be installed
    } else {
        Log "Drivers OK." "Green"
    }

    return $true
}

# ---------------- MAIN ----------------

$PSScriptRoot = Split-Path -Parent $MyInvocation.MyCommand.Definition
$driversPath  = Join-Path $PSScriptRoot "drivers"
$toolExe      = Join-Path $PSScriptRoot "tools\esptool.exe"

$binBoot     = Join-Path $PSScriptRoot "ada\ada_1_7_2.ino.bootloader.bin"
$binParts    = Join-Path $PSScriptRoot "ada\ada_1_7_2.ino.partitions.bin"
$binBootApp  = Join-Path $PSScriptRoot "ada\boot_app0.bin"
$binFirmware = Join-Path $PSScriptRoot "ada\ada_1_7_2.ino.bin"

if (-not (Test-Path $toolExe)) { Log "ERROR: esptool not found (tools\esptool.exe)" "Red"; exit 2 }

Log "Searching for ADA..." "Cyan"
$ports = Get-AdaPorts

$didInstall = $false
$start = Get-Date

while ($ports.Count -eq 0) {
    $elapsed = (New-TimeSpan -Start $start -End (Get-Date)).TotalSeconds
    if ($elapsed -ge $MaxWaitSeconds) {
        Log "No ADA device found. Tip: Unplug/replug USB, then restart." "Red"
        exit 3
    }

    if (-not $didInstall) {
        # First attempt: auto driver install
        Install-Drivers $driversPath | Out-Null
        $didInstall = $true
        Start-Sleep -Seconds 2
    } else {
        # Driver already tried, just waiting/rescanning
        Log "Searching for device... (after 5-6 attempts, unplug and replug USB...)" "Yellow"
        Start-Sleep -Seconds $RetrySeconds
    }

    $ports = Get-AdaPorts
}

# Port found
$targetPort = $ports[0].Port
Log "Found: $($ports[0].Type) - $targetPort" "Green"

Log "Starting firmware upload... (do not unplug USB)" "Cyan"

& "$toolExe" --chip esp32 --port "$targetPort" --baud $BaudRate `
  --before default_reset --after hard_reset write_flash -z `
  --flash_mode keep --flash_freq keep --flash_size keep `
  0x1000 "$binBoot" 0x8000 "$binParts" 0xe000 "$binBootApp" 0x10000 "$binFirmware"

if ($LASTEXITCODE -ne 0) {
    Log "ERROR during upload (esptool exit: $LASTEXITCODE)" "Red"
    exit 4
}

Log "Update successful " "Green"

Write-Host ""
Write-Host "Done! Press Enter to exit..." -ForegroundColor Green
Read-Host | Out-Null

exit 0