# =============================================================================
# ADA P1 Meter - Auto Firmware Frissítő (CH340 / CP210x + auto driver install)
# =============================================================================

$ErrorActionPreference = "Continue"
$ProgressPreference = "SilentlyContinue"

# ---- Beállítások (ha akarod, itt finomíts) ----
$RetrySeconds   = 2      # ennyit vár két keresés között
$MaxWaitSeconds = 90     # ennyi ideig próbálkozik összesen, aztán kilép
$BaudRate       = 921600 # esptool baud

function Log($msg, $color="Gray") {
    Write-Host $msg -ForegroundColor $color
}

function Is-Admin {
    $id = [Security.Principal.WindowsIdentity]::GetCurrent()
    $p  = New-Object Security.Principal.WindowsPrincipal($id)
    return $p.IsInRole([Security.Principal.WindowsBuiltInRole]::Administrator)
}

# --- ADMIN KÉNYSZERÍTÉS (mindig) ---
if (-not (Is-Admin)) {
    Log "Admin jogosultsag kell, ujrainditom..." "Yellow"
    $ps = (Get-Process -Id $PID).Path
    $script = $MyInvocation.MyCommand.Path
    Start-Process -FilePath $ps -ArgumentList "-NoProfile -ExecutionPolicy Bypass -File `"$script`"" -Verb RunAs | Out-Null
    exit
}

function Get-AdaPorts {
    # CH340: VID_1A86, CP210x: VID_10C4
    $vidRegex = 'VID_1A86|VID_10C4'

    $devs = Get-CimInstance Win32_PnPEntity -ErrorAction SilentlyContinue |
        Where-Object { $_.PNPDeviceID -match $vidRegex }

    $ports = @()

    foreach ($d in $devs) {
        $regPath = "HKLM:\SYSTEM\CurrentControlSet\Enum\$($d.PNPDeviceID)\Device Parameters"
        $portName = $null

        try {
            $portName = (Get-ItemProperty -Path $regPath -Name "PortName" -ErrorAction Stop).PortName
        } catch {
            continue
        }

        if ($portName -notmatch '^COM\d+$') { continue }

        $type = if ($d.PNPDeviceID -match 'VID_1A86') { 'CH340' } elseif ($d.PNPDeviceID -match 'VID_10C4') { 'CP210x' } else { 'USB-UART' }

        $ports += [PSCustomObject]@{
            Port = $portName
            Name = $d.Name
            Type = $type
            PnpId = $d.PNPDeviceID
        }
    }

    $ports | Sort-Object Port -Unique
}

function Install-Drivers($DriversRoot) {
    $pnputil = Join-Path $env:SystemRoot "System32\pnputil.exe"
    if (-not (Test-Path $pnputil)) { Log "HIBA: pnputil nem talalhato" "Red"; return $false }
    if (-not (Test-Path $DriversRoot)) { Log "HIBA: drivers mappa nem talalhato" "Red"; return $false }

    $infFiles = Get-ChildItem -Path $DriversRoot -Filter "*.inf" -Recurse -File -ErrorAction SilentlyContinue
    if (-not $infFiles -or $infFiles.Count -eq 0) { Log "HIBA: nincs INF a drivers mappaban" "Red"; return $false }

    Log "Driver telepites..." "Yellow"

    $anyOk = $false
    foreach ($inf in $infFiles) {
        # Csendesebb futás: kimenetet eldobjuk, csak hibakódot figyelünk
        & $pnputil /add-driver "$($inf.FullName)" /install | Out-Null
        if ($LASTEXITCODE -eq 0) { $anyOk = $true }
    }

    # eszközök újraszkennelése
    & $pnputil /scan-devices | Out-Null

    if (-not $anyOk) {
        Log "Driver telepites lefutott, de nem latok sikeres telepitest (alairas/kompatibilitas lehet)." "Yellow"
        # ettől még lehet, hogy már korábban fent volt, szóval folytatjuk
    } else {
        Log "Driver OK." "Green"
    }

    return $true
}

# ---------------- MAIN ----------------

$PSScriptRoot = Split-Path -Parent $MyInvocation.MyCommand.Definition
$driversPath  = Join-Path $PSScriptRoot "drivers"
$toolExe      = Join-Path $PSScriptRoot "tools\esptool.exe"

$binBoot     = Join-Path $PSScriptRoot "ada\ada_1_7_2.ino.bootloader.bin"
$binParts    = Join-Path $PSScriptRoot "ada\ada_1_7_2.ino.partitions.bin"
$binBootApp  = Join-Path $PSScriptRoot "ada\boot_app0.bin"
$binFirmware = Join-Path $PSScriptRoot "ada\ada_1_7_2.ino.bin"

if (-not (Test-Path $toolExe)) { Log "HIBA: esptool nincs meg (tools\esptool.exe)" "Red"; exit 2 }

Log "ADA keresese..." "Cyan"
$ports = Get-AdaPorts

$didInstall = $false
$start = Get-Date

while ($ports.Count -eq 0) {
    $elapsed = (New-TimeSpan -Start $start -End (Get-Date)).TotalSeconds
    if ($elapsed -ge $MaxWaitSeconds) {
        Log "Nem talalok ADA eszkozt. Lehet hogy ez nem USB adatkabel? Tipp: dugd ki/be az USB-t, majd inditsd ujra." "Red"
        exit 3
    }

    if (-not $didInstall) {
        # első körben automatikusan driver telepítés
        Install-Drivers $driversPath | Out-Null
        $didInstall = $true
        Start-Sleep -Seconds 2
    } else {
        # már próbáltunk drivert, itt már csak várunk / újrakeresünk
        Log "Kereses... (10 probalkozas utan huzd ki az USB-t es dugd vissza...)" "Yellow"
        Start-Sleep -Seconds $RetrySeconds
    }

    $ports = Get-AdaPorts
}

# találtunk portot
$targetPort = $ports[0].Port
Log "Talaltam: $($ports[0].Type) - $targetPort" "Green"

Log "Firmware feltoltes indul... (ne huzd ki az USB-t)" "Cyan"

& "$toolExe" --chip esp32 --port "$targetPort" --baud $BaudRate `
  --before default_reset --after hard_reset write_flash -z `
  --flash_mode keep --flash_freq keep --flash_size keep `
  0x1000 "$binBoot" 0x8000 "$binParts" 0xe000 "$binBootApp" 0x10000 "$binFirmware"

if ($LASTEXITCODE -ne 0) {
    Log "HIBA a feltoltes soran (esptool exit: $LASTEXITCODE)" "Red"
    exit 4
}

Log "Sikeres frissites " "Green"

Write-Host ""
Write-Host "Kesz! Nyomj Enter-t a kilepeshez..." -ForegroundColor Green
Read-Host | Out-Null

exit 0
